// ----------------------------------------------------------------------------------
// Microsoft Developer & Platform Evangelism
// 
// Copyright (c) Microsoft Corporation. All rights reserved.
// 
// THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, 
// EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
// OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
// ----------------------------------------------------------------------------------
// The example companies, organizations, products, domain names,
// e-mail addresses, logos, people, places, and events depicted
// herein are fictitious.  No association with any real company,
// organization, product, domain name, email address, logo, person,
// places, or events is intended or should be inferred.
// ----------------------------------------------------------------------------------

namespace FedMetadataClient
{
    using System;
    using System.Collections.Specialized;
    using System.Configuration;
    using System.IO;
    using System.Linq;
    using System.Net;
    using System.Text;
    using System.Web;
    using System.Xml;
    using System.Xml.Serialization;

    public class Program
    {
        static void Main(string[] args)
        {
            string stsBaseAddress = ConfigurationManager.AppSettings["stsBaseAddress"];
            string stsPath = ConfigurationManager.AppSettings["stsPath"];
            string serviceNamespace = ConfigurationManager.AppSettings["serviceNamespace"];
            string acsHostName = ConfigurationManager.AppSettings["acsHostName"];
            string mgmtKey = ConfigurationManager.AppSettings["mgmtKey"];

            string stsAddress = string.Format("https://{0}/{1}", stsBaseAddress, stsPath);
            string acsSTSAddress = string.Format("https://{0}.{1}/WRAPv0.8", serviceNamespace, acsHostName);
            string fedMetadataAddress = string.Format("https://{0}/FederationMetadata/2007-06/FederationMetadata.xml", stsBaseAddress);

            // get the fed metadata from the STS
            string fedMetadata = GetFedMetadata(fedMetadataAddress);

            // create the issuer
            Issuer issuer = new Issuer();
            issuer.DisplayName = serviceNamespace + "SAML";
            issuer.FederationMetadata200706 = fedMetadata;

            // serialize the issuer
            string issuerBody = SerializeIssuer(issuer);

            // get a management token to authorize the mgmt operation
            string mgmtToken = GetMgmtTokenForIssuerOperation(serviceNamespace, mgmtKey, acsHostName);

            // create the issuer in ACS    
            WebClient mgmtClient = new WebClient();
            mgmtClient.BaseAddress = string.Format("https://{0}.{1}", serviceNamespace, acsHostName);

            mgmtClient.Headers.Add(HttpRequestHeader.Authorization, "WRAPv0.8" + " " + HttpUtility.UrlDecode(mgmtToken));
            mgmtClient.Headers.Add(HttpRequestHeader.ContentType, "application/xml");

            Byte[] responseBytes = mgmtClient.UploadData("mgmt/issuers", "POST", Encoding.UTF8.GetBytes(issuerBody));
            string tokenresponse = Encoding.UTF8.GetString(responseBytes);
        }

        private static string SerializeIssuer(Issuer issuer)
        {
            XmlSerializer serializer = new XmlSerializer(typeof(Issuer));
            StringBuilder sb = new StringBuilder(256);
            XmlWriterSettings writerSettings = new XmlWriterSettings();
            writerSettings.Indent = true;
            writerSettings.IndentChars = " ";
            writerSettings.OmitXmlDeclaration = true;

            using (XmlWriter writer = XmlTextWriter.Create(sb, writerSettings))
            {
                serializer.Serialize(writer, issuer);
            }

            return sb.ToString();
        }

        private static string GetFedMetadata(string fedMetadataAddress)
        {
            HttpWebRequest request = (HttpWebRequest)WebRequest.Create(fedMetadataAddress);
            request.Method = "GET";
            request.ContentLength = 0;

            HttpWebResponse response = (HttpWebResponse)request.GetResponse();
            Stream responseStream = response.GetResponseStream();

            return new StreamReader(responseStream).ReadToEnd();
        }

        private static string GetMgmtTokenForIssuerOperation(string serviceNamespace, string mgmtKey, string acsBaseAddress)
        {
            WebClient client = new WebClient();
            client.BaseAddress = string.Format("https://{0}-mgmt.{1}", serviceNamespace, acsBaseAddress);

            NameValueCollection values = new NameValueCollection();
            values.Add("wrap_name", "owner");
            values.Add("wrap_password", mgmtKey);
            values.Add("applies_to", string.Format("https://{0}.{1}/mgmt/issuers", serviceNamespace, acsBaseAddress));

            byte[] responseBytes = client.UploadValues("WRAPv0.8", values);
            string response = Encoding.UTF8.GetString(responseBytes);

            return response
               .Split('&')
               .Single(value => value.StartsWith("wrap_token=", StringComparison.OrdinalIgnoreCase))
               .Split('=')[1];
        }
    }
}